%
% function [Measures] = Whiteness_Measures_C(noise, opt)
%
% Equivalent to function "Whiteness_Measures", but computes local measures faster. 
% Uses a function, implemented in C, for extracting the blocks of an image: 
%           function "im2colstep", computed by Ron Rubinstein,
%           downloaded from: http://www.cs.technion.ac.il/~ronrubin/software.html
%
% This function computes, for the image input variable "noise", the six measures of whiteness defined in [1,2].
%
% [1] M. S. C. Almeida, M. A. T. Figueiredo, "New stopping criteria for iterative blind image deblurring based on
%     residual whiteness measures", IEEE Statistical Signal Processing Workshop - SSP'2011, Nice, France, 2011.
% [2] M. S. C. Almeida, M. A. T. Figueiredo, "Parameter Estimation for Blind and Non-Blind Deblurring Using Residual
%     Whiteness Measures", IEEE Trans. on Image Processing, 2013 (accepted).
%
% Outputs: six measures of whiteness (three global and three local):
%           1) Measures.R   -  measure based on covariance  (M_R in the paper)
%           2) Measures.RG  -  measure based on weighted covariance  (R_{RW} in the paper)
%           3) Measures.H   -  measure based on entropy  (M_H in the paper)
%           4) Measures.Rb  -  measure based on block covariance  (M_R^b in the paper)
%           5) Measures.RGb -  measure based on block weighted covariance  (R_{RW}^b in the paper)
%           6) Measures.Hb  -  measure based on block entropy  (M_H^b in the paper)
%
% Inputs:  noise - variable to measure the whiteness
%          opt   - method's options:
%           opt.find_local - control parameter that allows to
%                   turn of the computation of the local measures.
%                   If "0" does not computes local measures.
%                   If "1" (or different than 0) computes local measures.
%                   Default value = 1.
%           opt.szg  - Parameter that defines:
%                   1) the Gaussian weight (gausswin(szg*2-1)).
%                   2) the block size, in local measures
%                      (the block size is given by (sz_bloco-1)/2).
%                   Default = 5, not recommended to change.
%           opt.passo -  distance between blocks, for local measures.
%                   Default value = szg (i.e. 5 pixels).
%
%
% Mariana S. C. Almeida
% Instituto de Telecomunicaes, Lisbon, Portugal 
% marianascalmeida@gmail.com


function [Measures] = Whiteness_Measures_C(noise, opt)

try opt.find_local;     find_local=opt.find_local;      catch me, find_local = 1; end
try opt.szg;            szg = opt.szg;                  catch me, szg = 5; end
try opt.passo;          passo = opt.passo;              catch me, passo = szg; end

sz_bloco = 2*szg - 1;  % block size (9 by default)
lado = (sz_bloco-1)/2; % (4 by default)
Gaussmask = gausswin(szg*2-1)*gausswin(szg*2-1)'; % Gaussian window used for computing the weighted covariance measure

noise_00_int = noise(1+szg:end-szg,1+szg:end-szg);    % interior of noise, used to ignore border effects due to shifts within [-szg , szg]
noise_00_int = noise_00_int - mean(noise_00_int(:));  % zero mean
noise_00_int = noise_00_int/norm(noise_00_int,'fro'); % unit variance


for i=-szg+1:szg-1  % for all shifts in [-szg , szg] - first dimension
    for j=-szg+1:szg-1 % for all shifts in [-szg , szg] - second dimension
        
        noise_ij = (circshift(noise,[i j]));  % shift the noise
        noise_ij_int = noise_ij(1+szg:end-szg,1+szg:end-szg); % extract the interior (to avoid invalid borders)
        
        noise_ij_int = noise_ij_int-mean(noise_ij_int(:));    % zero mean
        noise_ij_int = noise_ij_int/norm(noise_ij_int,'fro'); % constant variance (1/sz_bloco^2)
        noise_prod = (noise_ij_int.*noise_00_int);
        R(szg+i,szg+j) = sum(sum(noise_prod));     % correlation (or covariance) at position (i,j)
        
        if find_local   % computations for local measures
            
            BLOCOS_00 = reshape(  im2colstep( noise_00_int, [sz_bloco sz_bloco] ,  [passo passo] )  , sz_bloco,sz_bloco,[] );
            BLOCOS_00 = BLOCOS_00 -  repmat( (mean(mean(BLOCOS_00))) , [sz_bloco sz_bloco 1] ); %zero mean
            BLOCOS_00 = BLOCOS_00 ./ repmat( sqrt(sum(sum(BLOCOS_00.*BLOCOS_00))) , [sz_bloco sz_bloco 1] ); % constant variance (1/sz_bloco^2)
            BLOCOS_ij = reshape(  im2colstep( noise_ij_int, [sz_bloco sz_bloco] ,  [passo passo] )  , sz_bloco,sz_bloco,[] );
            BLOCOS_ij = BLOCOS_ij -  repmat( (mean(mean(BLOCOS_ij))) , [sz_bloco sz_bloco 1] ); %zero mean
            BLOCOS_00 = BLOCOS_00 ./ repmat( sqrt(sum(sum(BLOCOS_ij.*BLOCOS_ij))) , [sz_bloco sz_bloco 1] ); % constant variance (1/sz_bloco^2)
            PI_BLOCOS(szg+i,szg+j,:) = sum(sum(BLOCOS_00.*BLOCOS_ij));
            Gaussmask_BLOCOS(szg+i,szg+j,:) = repmat(Gaussmask(szg+i,szg+j), [1 size(PI_BLOCOS,3)] );  % mark for Gaussian weighting of block covariances
            
        end % end of computation for local measures
        
    end
end
R = R/R(szg,szg);
Delta = 0*R; Delta(szg,szg)=1;
Measures.R = - norm( (Delta-R),'fro' ).^2;   %  Measure.R - covariance measure
Measures.RG = - norm( sqrt(Gaussmask).*(Delta-R),'fro').^2;  %  Measure.RG - weighted covariance measure

R_f_norm = abs(fft2(R))./ sum(sum(abs(fft2(R))));  %  fft2 and normalization to sum 1 (as a probability)
Measures.H = sum(sum(-R_f_norm.*log(R_f_norm)));   %  Measure.H - entropy measure

if find_local   % computations for local measures
    Delta_BLOCOS = zeros(size(PI_BLOCOS));
    Delta_BLOCOS(szg,szg,:) = PI_BLOCOS(szg,szg,:);
    Measures.Rb = - norm( (Delta_BLOCOS(:)-PI_BLOCOS(:)) ,'fro').^2/(size(PI_BLOCOS,3));  % Measure.Rb - block covariance measure
    Measures.RGb = - norm( sqrt(Gaussmask_BLOCOS(:)).*(Delta_BLOCOS(:)-PI_BLOCOS(:)) ,'fro').^2/(size(PI_BLOCOS,3)); % Measure.Rb - weighted block covariance measure
    
   %  Measure.Hb - block entropy measure:
    V_EntropBLOCOS=[];
    for nb=1:size(PI_BLOCOS,3)
        R_f_BLOCOS_n1n2 = fft2( squeeze(PI_BLOCOS(:,:,nb)) );
        R_f_BLOCOS_norm_n1n2 = abs(R_f_BLOCOS_n1n2)./ sum(sum(abs(R_f_BLOCOS_n1n2)));  % normalization - such as a probability
        V_EntropBLOCOS = [V_EntropBLOCOS sum(sum(-R_f_BLOCOS_norm_n1n2.*log(R_f_BLOCOS_norm_n1n2)))]; % Shannons entropy
    end
     Measures.Hb = mean(V_EntropBLOCOS); % Measure.Hb - block entropy measure
else
    Measures.Rb =[];
    Measures.RWb =[];
    Measures.Hb =[];
end




