%
% function [Measures] = Whiteness_Measures(noise, opt)
%
% Function that computes, for the image input variable "noise", the six measures of whiteness defined in [1,2].
% Function "Whiteness_Measures_C" is equivalent to this function, but computes local measures faster. 
%
% [1] M. S. C. Almeida, M. A. T. Figueiredo, "New stopping criteria for iterative blind image deblurring based on
%     residual whiteness measures", IEEE Statistical Signal Processing Workshop - SSP'2011, Nice, France, 2011.
% [2] M. S. C. Almeida, M. A. T. Figueiredo, "Parameter Estimation for Blind and Non-Blind Deblurring Using Residual
%     Whiteness Measures", IEEE Trans. on Image Processing, 2013 (accepted).
%
% Outputs: six measures of whiteness (three global and three local):
%           1) Measures.R   -  measure based on covariance  (M_R in the paper)
%           2) Measures.RG  -  measure based on weighted covariance  (R_{RW} in the paper)
%           3) Measures.H   -  measure based on entropy  (M_H in the paper)
%           4) Measures.Rb  -  measure based on block covariance  (M_R^b in the paper)
%           5) Measures.RGb -  measure based on block weighted covariance  (R_{RW}^b in the paper)
%           6) Measures.Hb  -  measure based on block entropy  (M_H^b in the paper)
%
% Inputs:  noise - variable to measure the whiteness
%          opt   - method's options:
%           opt.find_local - control parameter that allows to
%                   turn of the computation of the local measures.
%                   If "0" does not computes local measures.
%                   If "1" (or different than 0) computes local measures.
%                   Default value = 1.
%           opt.szg  - Parameter that defines:
%                   1) the Gaussian weight (gausswin(szg*2-1)).
%                   2) the block size, in local measures
%                      (the block size is given by (sz_bloco-1)/2).
%                   Default = 5, not recommended to change.
%           opt.passo -  distance between blocks, for local measures.
%                   Default value = szg (i.e. 5 pixels).
%
%
% Mariana S. C. Almeida
% Instituto de Telecomunicaes, Lisbon, Portugal 
% marianascalmeida@gmail.com


function [Measures] = Whiteness_Measures(noise, opt)

try opt.find_local;     find_local=opt.find_local;      catch me, find_local = 1; end
try opt.szg;            szg = opt.szg;                  catch me, szg = 5; end
try opt.passo;          passo = opt.passo;              catch me, passo = szg; end

sz_bloco = 2*szg - 1;  % block size (9 by default)
lado = (sz_bloco-1)/2; % (4 by default)
Gaussmask = gausswin(szg*2-1)*gausswin(szg*2-1)'; % Gaussian window used for computing the weighted covariance measure

noise_00_int = noise(1+szg:end-szg,1+szg:end-szg);    % interior of noise, used to ignore border effects due to shifts within [-szg , szg]
noise_00_int = noise_00_int - mean(noise_00_int(:));  % zero mean
noise_00_int = noise_00_int/norm(noise_00_int,'fro'); % unit variance


for i=-szg+1:szg-1  % for all shifts in [-szg , szg] - first dimension
    for j=-szg+1:szg-1 % for all shifts in [-szg , szg] - second dimension
        
        noise_ij = (circshift(noise,[i j]));  % shift the noise
        noise_ij_int = noise_ij(1+szg:end-szg,1+szg:end-szg); % extract the interior (to avoid invalid borders)
        
        noise_ij_int = noise_ij_int-mean(noise_ij_int(:));    % zero mean
        noise_ij_int = noise_ij_int/norm(noise_ij_int,'fro'); % constant variance (1/sz_bloco^2)
        noise_prod = (noise_ij_int.*noise_00_int);
        R(szg+i,szg+j) = sum(sum(noise_prod));     % correlation (or covariance) at position (i,j)
       
        if find_local   % computations for local measures

            nk1 = 1;
            for k1 = 1+lado+szg : passo : size(noise_ij,1)-lado-szg,  % loop on blocks - first dimension
                nk2 = 1;
                for k2 = 1+lado+szg: passo : size(noise_ij,2)-lado-szg, % loop on blocks - second dimension
                    bloco_00 = noise(k1-lado:k1+lado,k2-lado:k2+lado);  % centered block
                    bloco_00 = bloco_00 - mean(bloco_00(:));            % zero mean
                    bloco_00 = bloco_00/norm(bloco_00(:),'fro');        % constant variance (1/sz_bloco^2)
                    bloco_00 = bloco_00/sqrt(sum(bloco_00(:).^2));        % variance
                    bloco_ij = noise_ij(k1-lado:k1+lado,k2-lado:k2+lado); % block at position i,j
                    bloco_ij = bloco_ij - mean(bloco_ij(:));              % zero mean
                    bloco_ij = bloco_ij/norm(bloco_ij(:),'fro');          % constant variance (1/sz_bloco^2)
                    PI_bloco(nk1,nk2,szg+i,szg+j) = sum(bloco_00(:).*bloco_ij(:)); % covariance of block nk1 nk2 at position (i,j)
                    nk2=nk2+1;
                   end
                nk1=nk1+1;
            end
            Gaussmask_bloco(:,:,szg+i,szg+j) = repmat(Gaussmask(szg+i,szg+j), size(PI_bloco,1), size(PI_bloco,2));  % mark for Gaussian weighting of block covariances
        end % end of computation for local measures
        
    end
end
R = R/R(szg,szg);
Delta = 0*R; Delta(szg,szg)=1;
Measures.R = - norm( (Delta-R),'fro' ).^2;   %  Measure.R - covariance measure
Measures.RG = - norm( sqrt(Gaussmask).*(Delta-R),'fro').^2;  %  Measure.RG - weighted covariance measure

R_f_norm = abs(fft2(R))./ sum(sum(abs(fft2(R))));  %  fft2 and normalization to sum 1 (as a probability)
Measures.H = sum(sum(-R_f_norm.*log(R_f_norm)));   %  Measure.H - entropy measure

if find_local   % computations for local measures
    Delta_bloco = zeros(size(PI_bloco));
    Delta_bloco(:,:,szg,szg) = PI_bloco(:,:,szg,szg);
    Measures.Rb = - norm( (Delta_bloco(:)-PI_bloco(:)) ,'fro').^2/((nk1-1)*(nk2-1));  % Measure.Rb - block covariance measure
    Measures.RGb = - norm( sqrt(Gaussmask_bloco(:)).*(Delta_bloco(:)-PI_bloco(:)) ,'fro').^2/((nk1-1)*(nk2-1)); % Measure.Rb - weighted block covariance measure
    
    %  Measure.Hb - block entropy measure:
    V_EntropBloco=[];
    for n1=1:size(PI_bloco,1)   % loop over blocks - first dimension
        for n2=1:size(PI_bloco,2)   % loop over blocks - second dimension
            R_f_bloco_n1n2 = fft2( squeeze(PI_bloco(n1,n2,:,:)) );
            R_f_bloco_norm_n1n2 = abs(R_f_bloco_n1n2)./ sum(sum(abs(R_f_bloco_n1n2)));  % normalization - such as a probability
            V_EntropBloco = [V_EntropBloco sum(sum(-R_f_bloco_norm_n1n2.*log(R_f_bloco_norm_n1n2)))]; % Shannons entropy
        end
    end
    Measures.Hb = mean(V_EntropBloco); % Measure.Hb - block entropy measure
    
else
    Measures.Rb =[];
    Measures.RWb =[];
    Measures.Hb =[];
end




