% DemoWiteness:  Demo of the stopping criteria proposed in [1] [2]. 
%                Demo applied to a result of the BID method of [3].    
%                Calls funciton "Whiteness_Measures_C", which returns the six measures proposed in [1] [2].
%                Function "Whiteness_Measures", which does not uses the mex function "im2colstep" 
%                (by Ron Rubinstein; http://www.cs.technion.ac.il/~ronrubin/software.html),
%                can be called alternatively to "Whiteness_Measures_C".
%
% [1] M. S. C. Almeida, M. A. T. Figueiredo, "New stopping criteria for iterative blind image deblurring based on
%     residual whiteness measures", IEEE Statistical Signal Processing Workshop - SSP'2011, Nice, France, 2011.
% [2] M. S. C. Almeida, M. A. T. Figueiredo, "Parameter Estimation for Blind and Non-Blind Deblurring Using Residual
%     Whiteness Measures", IEEE Trans. on Image Processing, 2013 (accepted).
%
%
% Mariana S. C. Almeida
% Instituto de Telecomunicaes, Lisbon, Portugal 
% marianascalmeida@gmail.com
%
 
clear all, 
close all 
drawnow
 
load Camera9x9_30dB   % Loads the results obtained using the BID method of [3] for deblurring
                    % cameraman image, degraded with a 9x9 uniform blur, at 30dB of BSNR.
                    % [3]  M. S. C. Almeida and L. B. Almeida, "Blind and Semi-Blind Deblurring of Natural Images",
                    % IEEE Trans. Image Processing, Vol.19, pp. 36-52, January, 2010. 
 
 
[sz1 sz2 nit] = size(XX); % XX - vector with the image estimates
[h_sz1 h_sz2 auxi]  = size(HH_support); % HH_support - vector with the filter estimates
[ysz1 ysz2 ] = size(y); % y - degraded image.
 
side1 = (h_sz1-1)/2;
side2 = (h_sz2-1)/2;

fprintf('Computing measures of whiteness:')
for it = 2:nit 
    
    x = squeeze(XX(:,:,it)); % image estimate at iteration "it"
    ht = squeeze(HH_support(:,:,it-1));   % filter (within the limited support) that was used to estimate x
    h = ht2h(ht, size(x,1) , size(x,2) ); % filter with the size of x and centered at the spectral origin: pixel (1,1).
    
    hx = ifft2(fft2(h).*fft2(x));                  % Hx estimate, assuming circular boundary conditions.
    hx = hx(1+side1:end-side1, 1+side2:end-side2); % extraction of the pixels with valid boundary conditions.
    [hx_sz1 hx_sz2]  = size(hx);
    % residual estimate:
    if ysz1 > hx_sz1
        extra = (ysz1-sz1)/2;
        noise_it = y(1+extra:end-extra,1+extra:end-extra) - hx;  
    elseif hx_sz1 > ysz1
        extra = (sz1-ysz1)/2;
        noise_it = y - hx(1+extra:end-extra,1+extra:end-extra);
    else; noise_it = y - hx;
    end
 
   % options for function Whiteness_Measures_C
    opt.find_local = 1; % opt.find_local = 1; computes global and local measures
                        % opt.find_local = 0; only computes global measures
    [Measures] = Whiteness_Measures_C(noise_it, opt);    
    
                          %  Measures of whiteness, based on:
    R(it) = Measures.R;   %  covariance (M_R in the paper)
    RG(it) = Measures.RG; %  weighted covariance  (M_RG in the paper)
    H(it) = Measures.H;   %  entropy measure (M_H in the paper)
    if opt.find_local
        Rb(it) = Measures.Rb;   %  block covariance  (M_R^b in the paper)
        RGb(it) = Measures.RGb; %  block weighted covariance  (M_RW^b in the paper)
        Hb(it) = Measures.Hb;   %  block entropy  (M_H^b in the paper)
    end
    
    NOISE(:,:,it)=noise_it;
    
    fprintf('.')
end
 
R(1)=R(2); RG(1)=RG(2); H(1)=H(2);
% Best iteration according to ISNR and the whiteness measures:
[maxi it_isnr1] = max(ISNR_res1);
[maxi it_isnr3] = max(ISNR_res3);
[maxi it_R] = max(R);  
[maxi it_RG] = max(RG);  
[maxi it_H] = max(H); 
if opt.find_local
    Rb(1)=Rb(2); RGb(1)=RGb(2); Hb(1)=Hb(2);
    [maxi it_Rb] = max(Rb);
    [maxi it_RGb] = max(RGb);
    [maxi it_Hb] = max(Hb);
end
 
% PRINTS the best iterations based on various criteria:
fprintf('\ncriterion:   ISNR_r1  ISNR_r3     R    RG    H')
if opt.find_local, fprintf('     Rb   RGb   Hb '), end
fprintf('\niteration:     %d        %d       %d   %d    %d' , it_isnr1,it_isnr3,it_R,it_RG,it_H);
if opt.find_local, fprintf('    %d   %d    %d' , it_Rb,it_RGb,it_Hb ), end
fprintf('\n');
 
% PLOT residual estimates at various iterations  
scrsz = get(0,'ScreenSize');
figure('Position',[scrsz(3)/6 scrsz(4)*1/3 scrsz(4)/2 scrsz(4)/2])
subplot(2,2,1), imagesc(NOISE(:,:,5)),   colormap gray, axis image, title('residual at it=5')
subplot(2,2,2), imagesc(NOISE(:,:,it_RGb-6)),  colormap gray, axis image, title( strcat('residual at it=',num2str(it_RGb-6)) )
subplot(2,2,3), imagesc(NOISE(:,:,it_RGb)),  colormap gray, axis image, title(strcat('residual at it=',num2str(it_RGb)))
subplot(2,2,4), imagesc(NOISE(:,:,end)), colormap gray, axis image, title(strcat('residual at it=',num2str(size(NOISE,3))))
drawnow
 
% PLOT various measures along the iterations: isnr, H and RG
figure('Position',[scrsz(3)/6+scrsz(4)/2  scrsz(4)*1/3 scrsz(4)/2 scrsz(4)/2])
subplot(3,1,1), plot(ISNR_res3), colormap gray,  title('adapted ISNR-res3')
subplot(3,1,2), semilogy(H), colormap gray,  title('H')
subplot(3,1,3), semilogy(RG), colormap gray, title('RG')
drawnow
 
% PLOT image estimates, based on isnr and RGb
figure('Position',[scrsz(3)/6+scrsz(4)  scrsz(4)*1/10 scrsz(4)/3 scrsz(4)*3/4])
subplot(2,1,1), imagesc(XX(:,:,it_isnr3)), colormap gray, axis image, 
title(strcat('image estimate - it(ISNR-res3)=',num2str(it_isnr3)))
subplot(2,1,2), imagesc(XX(:,:,it_RGb)), colormap gray, axis image, 
title(strcat('image estimate - it(MGb)=',num2str(it_RGb)))
drawnow
 


